  /*
   *  Object %CRYS_RSA_BUILD.c    : %
   *  State           :  %state%
   *  Creation date   :  Wed Nov 17 17:25:55 2004
   *  Last modified   :  %modify_time%
   */
  /** @file
   *  \brief this module contains the routines used to create the public and private keys
   *         database and routines to check the validity of the private and public keys database. 
   *
   *  \version CRYS_RSA_BUILD.c#1:csrc:6
   *  \author adams
   *  \remarks Copyright (C) 2004 by Discretix Technologies Ltd.
   *           All Rights reserved
   */



/************* Include Files ****************/

/* .............. CRYS level includes ................. */

#include "DX_VOS_Mem.h"
#include "CRYS.h"
#include "CRYS_RSA_error.h"
#include "dx_hw_defs.h"
#include "crys_host_op_code.h"


 
/************************ Defines ******************************/

/* canceling the lint warning:
   Use of goto is deprecated */
/*lint --e{801} */

/************************ Enums ******************************/


/************************ Typedefs ******************************/


/************************ Global Data ******************************/

/************************ Public Functions ******************************/
#if !defined(_INTERNAL_CRYS_NO_RSA_ENCRYPT_SUPPORT) && !defined(_INTERNAL_CRYS_NO_RSA_VERIFY_SUPPORT)
/******************************************************************************************
   @brief CRYS_RSA_Build_PubKey populates a CRYSRSAPubKey_t structure with
   the provided modulus and exponent.
   
   assumption : the modulus and the exponent are presented in big endian.
   
   @param[out] PubKey_ptr - a pointer to the public key structure. This structure will be used as
                           an input to the CRYS_RSA_PRIM_Encrypt API.
 
   @param[in] Exponent_ptr - a pointer to the exponent stream of bytes ( Big endian ).
   @param[in] ExponentSize - The size of the exponent in bytes.  
   @param[in] Modulus_ptr  - a pointer to the modulus stream of bytes ( Big endian ) the MS bit must be set to '1'.
   @param[in] ModulusSize  - The size of the modulus in bytes. Size supported are:   
							   - for PKI without PKA: HW: all 32 bytes multiples between 64 - 256;
							   - for PKI without PKA: HW all 4 bytes multiples between 64 - 264;
*/
CEXPORT_C CRYSError_t CRYS_RSA_Build_PubKey(  CRYS_RSAUserPubKey_t *UserPubKey_ptr,
				                    DxUint8_t *Exponent_ptr,
				                    DxUint16_t ExponentSize,
				                    DxUint8_t *Modulus_ptr,
				                    DxUint16_t ModulusSize )
{
 	return DX_HW_Dispatcher_ptr( DX_SEP_HOST_SEP_PROTOCOL_HOST_RSA_BUILD_PUB_KEY_OP_CODE, 5,
	   							 /* PARAMETERS: */
                                 UserPubKey_ptr, 
								 Exponent_ptr, ExponentSize,
								 Modulus_ptr, ModulusSize );

}/* END OF CRYS_RSA_Build_PubKey */

#endif /*!defined(_INTERNAL_CRYS_NO_RSA_ENCRYPT_SUPPORT) && !defined(_INTERNAL_CRYS_NO_RSA_VERIFY_SUPPORT)*/

#if !defined(_INTERNAL_CRYS_NO_RSA_DECRYPT_SUPPORT) && !defined(_INTERNAL_CRYS_NO_RSA_SIGN_SUPPORT)
/******************************************************************************************
   @brief CRYS_RSA_Build_PrivKey populates a CRYSRSAPrivKey_t structure with
          the provided modulus and exponent, marking the key as a "non-CRT" key.

          Assumption : the modulus and the exponent are presented in big endian.
 
   @param[out] UserPrivKey_ptr - a pointer to the public key structure. this structure will be used as
                           an input to the CRYS_RSA_PRIM_Decrypt API.
 
   @param[in] PrivExponent_ptr - a pointer to the private exponent stream of bytes ( Big endian ).
   @param[in] PrivExponentSize - The size of the private exponent in bytes.
   @param[in] PubExponent_ptr - a pointer to the public exponent stream of bytes ( Big endian ).
   @param[in] PubExponentSize - The size of the public exponent in bytes.    
   @param[in] Modulus_ptr  - a pointer to the modulus stream of bytes ( Little endian ) the MS bit must be set to '1'.
   @param[in] ModulusSize  - The size of the modulus in bytes. Sizes supported are:   
								   - for PKI without PKA: HW: all 32 bytes multiples between 64 - 256;
								   - for PKI without PKA: HW all 4 bytes multiples between 64 - 264;
*/
CEXPORT_C CRYSError_t CRYS_RSA_Build_PrivKey(CRYS_RSAUserPrivKey_t *UserPrivKey_ptr,
				                   DxUint8_t             *PrivExponent_ptr,
				                   DxUint16_t             PrivExponentSize,
				                   DxUint8_t             *PubExponent_ptr,
				                   DxUint16_t             PubExponentSize,
				                   DxUint8_t             *Modulus_ptr,
				                   DxUint16_t             ModulusSize )
{				                   
 	return DX_HW_Dispatcher_ptr( DX_SEP_HOST_SEP_PROTOCOL_HOST_RSA_BUILD_PRIV_KEY_OP_CODE, 7,
	   							 /* PARAMETERS: */
								 UserPrivKey_ptr,
								 PrivExponent_ptr,
								 PrivExponentSize,
								 PubExponent_ptr,
								 PubExponentSize,
								 Modulus_ptr,
								 ModulusSize );

}/* END OF CRYS_RSA_Build_PrivKey */



/******************************************************************************************

   @brief CRYS_RSA_Build_PrivKey populates a CRYSRSAPrivKey_t structure with
          the provided parameters, marking the key as a "CRT" key.

   @param[out] UserPrivKey_ptr - a pointer to the public key structure. this structure will be used as
                            an input to the CRYS_RSA_PRIM_Decrypt API.
 
   @param[in] P_ptr - a pointer to the first factor stream of bytes ( Big endian ).
   @param[in] PSize - the size of the first factor in bits.
   @param[in] Q_ptr - a pointer to the second factor stream of bytes ( Big endian ).
   @param[in] QSize - the size of the second factor in bits.
   @param[in] dP_ptr - a pointer to the first factors CRT exponent stream of bytes ( Big endian ).
   @param[in] dPSize - the size of the first factors CRT exponent in bytes.
   @param[in] dQ_ptr - a pointer to the second factors CRT exponent stream of bytes ( Big endian ).
   @param[in] dQSize - the size of the second factors CRT exponent in bytes.
   @param[in] qInv_ptr - a pointer to the first CRT coefficient stream of bytes ( Big endian ).
   @param[in] dQSize - the size of the first CRT coefficient in bytes.

*/
CEXPORT_C CRYSError_t CRYS_RSA_Build_PrivKeyCRT(CRYS_RSAUserPrivKey_t *UserPrivKey_ptr,
				                      DxUint8_t *P_ptr, 
				                      DxUint16_t PSize,
				                      DxUint8_t *Q_ptr,
				                      DxUint16_t QSize,
				                      DxUint8_t *dP_ptr, 
				                      DxUint16_t dPSize,
				                      DxUint8_t *dQ_ptr,
				                      DxUint16_t dQSize,
				                      DxUint8_t *qInv_ptr,
				                      DxUint16_t qInvSize)
{				                   
 	return DX_HW_Dispatcher_ptr( DX_SEP_HOST_SEP_PROTOCOL_HOST_RSA_BUILD_PRIV_KEY_CRT_OP_CODE, 11,
	   							 /* PARAMETERS: */
								 UserPrivKey_ptr,
								 P_ptr, PSize,
								 Q_ptr, QSize,
								 dP_ptr, dPSize,
								 dQ_ptr, dQSize,
								 qInv_ptr, qInvSize );

}/* END OF CRYS_RSA_Build_PrivKeyCRT */

#endif /*!defined(_INTERNAL_CRYS_NO_RSA_DECRYPT_SUPPORT) && !defined(_INTERNAL_CRYS_NO_RSA_SIGN_SUPPORT)*/



#if !defined(_INTERNAL_CRYS_NO_RSA_ENCRYPT_SUPPORT) && !defined(_INTERNAL_CRYS_NO_RSA_VERIFY_SUPPORT)

/******************************************************************************************
   @brief CRYS_RSA_Get_PubKey gets the e,n public key from the database.
   
   @param[in] UserPubKey_ptr - A pointer to the public key structure. 
                               This structure is used as input to the CRYS_RSA_PRIM_Encrypt API.
 
   @param[out] Exponent_ptr - A pointer to the exponent stream of bytes (Big-Endian format)
   @param[in,out] ExponentSize_ptr - the size of the exponent buffer in bytes, it is updated to the 
                  actual size of the exponent, in bytes.  
   @param[out] Modulus_ptr  - A pointer to the modulus stream of bytes (Big-Endian format).
			   The MS (most significant) bit must be set to '1'.
   @param[in,out] ModulusSize_ptr  - the size of the modulus buffer in bytes, it is updated to the 
                  actual size of the modulus, in bytes.

   NOTE: All members of input UserPrivKey structure must be initialized, including public key
         e pointer and it size.

*/
CEXPORT_C CRYSError_t CRYS_RSA_Get_PubKey(  CRYS_RSAUserPubKey_t *UserPubKey_ptr,
				                  DxUint8_t  *Exponent_ptr,
				                  DxUint16_t *ExponentSize_ptr,
				                  DxUint8_t  *Modulus_ptr,
				                  DxUint16_t *ModulusSize_ptr )
{	
 	return DX_HW_Dispatcher_ptr( DX_SEP_HOST_SEP_PROTOCOL_HOST_RSA_GET_PUB_KEY_OP_CODE, 5,
	   							 /* PARAMETERS: */
								 UserPubKey_ptr,
								 Exponent_ptr, ExponentSize_ptr,
								 Modulus_ptr, ModulusSize_ptr );

}/* END OF CRYS_RSA_Get_PubKey */   

/******************************************************************************************
   @brief CRYS_RSA_Get_ModSizeFromPubKey extracts the modulus size from a given public key data structure.
   
   @param[in] UserPubKey_ptr - A pointer to the public key structure, as returned by
                               CRYS_RSA_Build_PubKey.
 
   @param[out] ModulusSize_ptr  -  The actual size of the modulus, in bytes.  
*/
CEXPORT_C CRYSError_t CRYS_RSA_Get_ModSizeFromPubKey(  CRYS_RSAUserPubKey_t *UserPubKey_ptr,
				                                     DxUint16_t *ModulusSize_ptr )
{	
 	return DX_HW_Dispatcher_ptr( DX_SEP_HOST_SEP_PROTOCOL_HOST_RSA_MOD_SIZE_OP_CODE, 5,
	   							 /* PARAMETERS: */
								 UserPubKey_ptr,
								 ModulusSize_ptr );

}/* END OF CRYS_RSA_Get_ModSizeFromPubKey */   
#endif /*!defined(CRYS_NO_RSA_ENCRYPT_SUPPORT) && !defined(_INTERNAL_CRYS_NO_RSA_VERIFY_SUPPORT)*/


#if !defined(_INTERNAL_CRYS_NO_RSA_DECRYPT_SUPPORT) && !defined(_INTERNAL_CRYS_NO_RSA_SIGN_SUPPORT)
/******************************************************************************************
   @brief CRYS_RSA_Get_PrivKey gets the D,n private key from the database.
   
   @param[in] UserPrivKey_ptr - A pointer to the private key structure. 
                               This structure is used as input to the CRYS_RSA_PRIM_Decrypt API.
 
   @param[out] PrivExponent_ptr - A pointer to the exponent stream of bytes (Big-Endian format)

   @param[in/out] PrivExponentSize - the size of the exponent buffer in bytes , it is updated to the 
                  actual size of the exponent, in bytes.  

   @param[out] PubExponent_ptr - a pointer to the public exponent stream of bytes ( Big endian ).
   
   @param[in/out] PubExponentSize - the size of the exponent buffer in bytes , it is updated to the 
                  actual size of the exponent, in bytes.    
   
   @param[out] Modulus_ptr  - A pointer to the modulus stream of bytes (Big-Endian format).
			   The MS (most significant) bit must be set to '1'.

   @param[in/out] ModulusSize_ptr  - the size of the modulus buffer in bytes , it is updated to the 
                  actual size of the modulus, in bytes.  
*/
CEXPORT_C CRYSError_t CRYS_RSA_Get_PrivKey(CRYS_RSAUserPrivKey_t *UserPrivKey_ptr,
				                   DxUint8_t             *PrivExponent_ptr,
				                   DxUint16_t            *PrivExponentSize_ptr,
				                   DxUint8_t             *PubExponent_ptr,
				                   DxUint16_t            *PubExponentSize_ptr,
				                   DxUint8_t             *Modulus_ptr,
				                   DxUint16_t            *ModulusSize_ptr )
{	
 	return DX_HW_Dispatcher_ptr( DX_SEP_HOST_SEP_PROTOCOL_HOST_RSA_GET_PRIV_KEY_OP_CODE, 7,
	   							 /* PARAMETERS: */
								 UserPrivKey_ptr,
								 PrivExponent_ptr,
								 PrivExponentSize_ptr,
								 PubExponent_ptr,
								 PubExponentSize_ptr,
								 Modulus_ptr,
								 ModulusSize_ptr );

}/* END OF CRYS_RSA_Get_PrivKey */   

/******************************************************************************************

   @brief CRYS_RSA_Get_PrivKeyCRT exports a CRYSRSAPrivKey_t structure data

   @param[In] UserPrivKey_ptr - a pointer to the public key structure. this structure will be used as
                            an input to the CRYS_RSA_PRIM_Decrypt API.
 
   @param[out] P_ptr - a pointer to the first factor stream of bytes ( Big endian ).
   @param[in/out] PSize_ptr - the size of the first factor buffer in bytes , it is updated to the 
                  actual size of the first factor, in bytes.
   @param[out] Q_ptr - a pointer to the second factor stream of bytes ( Big endian ).
   @param[in/out] QSize_ptr - the size of the second factor buffer in bytes , it is updated to the 
                  actual size of the second factor, in bytes.
   @param[out] dP_ptr - a pointer to the first factors CRT exponent stream of bytes ( Big endian ).
   @param[in/out] dPSize_ptr - the size of the first factor exponent buffer in bytes , it is updated to the 
                  actual size of the first factor exponent, in bytes.
   @param[out] dQ_ptr - a pointer to the second factors CRT exponent stream of bytes ( Big endian ).
   @param[in/out] dQSize_ptr - the size of the second factors CRT exponent buffer in bytes , it is updated to the 
                  actual size of the second factors CRT exponent, in bytes.
   @param[out] qInv_ptr - a pointer to the first CRT coefficient stream of bytes ( Big endian ).
   @param[in/out] qInvSize_ptr -  the size of the first CRT coefficient buffer in bytes , it is updated to the 
                  actual size of the first CRT coefficient, in bytes.
*/

CEXPORT_C CRYSError_t CRYS_RSA_Get_PrivKeyCRT(CRYS_RSAUserPrivKey_t *UserPrivKey_ptr,
				                      DxUint8_t *P_ptr, 
				                      DxUint16_t *PSize_ptr,
				                      DxUint8_t *Q_ptr,
				                      DxUint16_t *QSize_ptr,
				                      DxUint8_t *dP_ptr, 
				                      DxUint16_t *dPSize_ptr,
				                      DxUint8_t *dQ_ptr,
				                      DxUint16_t *dQSize_ptr,
				                      DxUint8_t *qInv_ptr,
				                      DxUint16_t *qInvSize_ptr)
{				                   
 	return DX_HW_Dispatcher_ptr( DX_SEP_HOST_SEP_PROTOCOL_HOST_RSA_GET_PRIV_KEY_CRT_OP_CODE, 11,
	   							 /* PARAMETERS: */
								 UserPrivKey_ptr,
								 P_ptr, PSize_ptr,
								 Q_ptr, QSize_ptr,
								 dP_ptr, dPSize_ptr,
							 	 dQ_ptr, dQSize_ptr,
								 qInv_ptr, qInvSize_ptr );

}/* END OF CRYS_RSA_Get_PrivKeyCRT */   


/******************************************************************************************

   @brief CRYS_RSA_Get_PrivKeyModulus export the modulus vector from CRYSRSAPrivKey_t structure.

   @param[out] UserPrivKey_ptr - a pointer to the private key structure. 
 
   @param[out] N_ptr - a pointer to the modulus vector of bytes ( Big endian ).
   @param[in/out] NSize_ptr - the size of the modulus buffer in bytes , it is updated to the 
                  actual size in bytes.
*/

CEXPORT_C CRYSError_t CRYS_RSA_Get_PrivKeyModulus(CRYS_RSAUserPrivKey_t *UserPrivKey_ptr,
				                      DxUint8_t *N_ptr, 
				                      DxUint16_t *NSize_ptr)
{				                   
 	return DX_HW_Dispatcher_ptr( DX_SEP_HOST_SEP_PROTOCOL_HOST_RSA_GET_PRIV_KEY_MODULUS_OP_CODE, 3,
	   							 /* PARAMETERS: */
								 UserPrivKey_ptr,
								 N_ptr, NSize_ptr );

}/* END OF CRYS_RSA_Get_PrivKeyCRT */   

#endif /*!defined(_INTERNAL_CRYS_NO_RSA_DECRYPT_SUPPORT) && !defined(_INTERNAL_CRYS_NO_RSA_SIGN_SUPPORT)*/

